"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var config_1 = require("./config");
exports.default = {
    meta: {
        docs: {
            category: 'emcascript6',
            description: 'prefer arrow functions',
            recommended: false,
        },
        fixable: 'code',
        schema: [
            {
                additionalProperties: false,
                properties: {
                    allowNamedFunctions: { type: 'boolean' },
                    classPropertiesAllowed: { type: 'boolean' },
                    disallowPrototype: { type: 'boolean' },
                    returnStyle: {
                        default: config_1.DEFAULT_OPTIONS.returnStyle,
                        pattern: '^(explicit|implicit|unchanged)$',
                        type: 'string',
                    },
                    singleReturnOnly: { type: 'boolean' },
                },
                type: 'object',
            },
        ],
    },
    create: function (context) {
        var options = context.options[0] || {};
        var getOption = function (name) {
            return typeof options[name] !== 'undefined'
                ? options[name]
                : config_1.DEFAULT_OPTIONS[name];
        };
        var allowNamedFunctions = getOption('allowNamedFunctions');
        var singleReturnOnly = getOption('singleReturnOnly');
        var classPropertiesAllowed = getOption('classPropertiesAllowed');
        var disallowPrototype = getOption('disallowPrototype');
        var returnStyle = getOption('returnStyle');
        var sourceCode = context.getSourceCode();
        var isBlockStatementWithSingleReturn = function (node) {
            return (node.body.body &&
                node.body.body.length === 1 &&
                node.body.body[0].type === 'ReturnStatement');
        };
        var isImplicitReturn = function (node) {
            return node.body && !node.body.body;
        };
        var returnsImmediately = function (node) {
            return isBlockStatementWithSingleReturn(node) || isImplicitReturn(node);
        };
        var getBodySource = function (node) {
            if (isBlockStatementWithSingleReturn(node) &&
                returnStyle !== 'explicit') {
                var returnValue = node.body.body[0].argument;
                var source = sourceCode.getText(returnValue);
                return returnValue.type === 'ObjectExpression' ? "(" + source + ")" : source;
            }
            if (isImplicitReturn(node) && returnStyle !== 'implicit') {
                return "{ return " + sourceCode.getText(node.body) + " }";
            }
            return sourceCode.getText(node.body);
        };
        var getParamsSource = function (params) {
            return params.map(function (param) { return sourceCode.getText(param); });
        };
        var getFunctionName = function (node) {
            return node && node.id && node.id.name ? node.id.name : '';
        };
        var isGenericFunction = function (node) { return Boolean(node.typeParameters); };
        var getGenericSource = function (node) { return sourceCode.getText(node.typeParameters); };
        var isAsyncFunction = function (node) { return node.async === true; };
        var isGeneratorFunction = function (node) { return node.generator === true; };
        var getReturnType = function (node) {
            var _a;
            return node.returnType &&
                node.returnType.range && (_a = sourceCode.getText()).substring.apply(_a, node.returnType.range);
        };
        var containsToken = function (type, value, node) {
            return sourceCode
                .getTokens(node)
                .some(function (token) { return token.type === type && token.value === value; });
        };
        var containsSuper = function (node) {
            return containsToken('Keyword', 'super', node);
        };
        var containsThis = function (node) {
            return containsToken('Keyword', 'this', node);
        };
        var containsArguments = function (node) {
            return containsToken('Identifier', 'arguments', node);
        };
        var containsTokenSequence = function (sequence, node) {
            return sourceCode.getTokens(node).some(function (_, tokenIndex, tokens) {
                return sequence.every(function (_a, i) {
                    var expectedType = _a[0], expectedValue = _a[1];
                    var actual = tokens[tokenIndex + i];
                    return (actual &&
                        actual.type === expectedType &&
                        actual.value === expectedValue);
                });
            });
        };
        var containsNewDotTarget = function (node) {
            return containsTokenSequence([
                ['Keyword', 'new'],
                ['Punctuator', '.'],
                ['Identifier', 'target'],
            ], node);
        };
        var writeArrowFunction = function (node) {
            var _a = getFunctionDescriptor(node), body = _a.body, isAsync = _a.isAsync, isGeneric = _a.isGeneric, generic = _a.generic, params = _a.params, returnType = _a.returnType;
            return 'ASYNC<GENERIC>(PARAMS)RETURN_TYPE => BODY'
                .replace('ASYNC', isAsync ? 'async ' : '')
                .replace('<GENERIC>', isGeneric ? generic : '')
                .replace('BODY', body)
                .replace('RETURN_TYPE', returnType ? returnType : '')
                .replace('PARAMS', params.join(', '));
        };
        var writeArrowConstant = function (node) {
            var name = getFunctionDescriptor(node).name;
            return 'const NAME = ARROW_FUNCTION'
                .replace('NAME', name)
                .replace('ARROW_FUNCTION', writeArrowFunction(node));
        };
        var getFunctionDescriptor = function (node) {
            return {
                body: getBodySource(node),
                isAsync: isAsyncFunction(node),
                isGenerator: isGeneratorFunction(node),
                isGeneric: isGenericFunction(node),
                name: getFunctionName(node),
                generic: getGenericSource(node),
                params: getParamsSource(node.params),
                returnType: getReturnType(node),
            };
        };
        var isPrototypeAssignment = function (node) {
            return context
                .getAncestors()
                .reverse()
                .some(function (ancestor) {
                var isPropertyOfReplacementPrototypeObject = ancestor.type === 'AssignmentExpression' &&
                    ancestor.left &&
                    ancestor.left.property &&
                    ancestor.left.property.name === 'prototype';
                var isMutationOfExistingPrototypeObject = ancestor.type === 'AssignmentExpression' &&
                    ancestor.left &&
                    ancestor.left.object &&
                    ancestor.left.object.property &&
                    ancestor.left.object.property.name === 'prototype';
                return (isPropertyOfReplacementPrototypeObject ||
                    isMutationOfExistingPrototypeObject);
            });
        };
        var isWithinClassBody = function (node) {
            return context
                .getAncestors()
                .reverse()
                .some(function (ancestor) {
                return ancestor.type === 'ClassBody';
            });
        };
        var isNamed = function (node) { return node.id && node.id.name; };
        var isNamedDefaultExport = function (node) {
            return isNamed(node) && node.parent.type === 'ExportDefaultDeclaration';
        };
        var isSafeTransformation = function (node) {
            return (!isGeneratorFunction(node) &&
                !containsThis(node) &&
                !containsSuper(node) &&
                !containsArguments(node) &&
                !containsNewDotTarget(node) &&
                (!isNamed(node) || !allowNamedFunctions) &&
                (!isPrototypeAssignment(node) || disallowPrototype) &&
                (!singleReturnOnly ||
                    (returnsImmediately(node) && !isNamedDefaultExport(node))));
        };
        var getMessage = function (node) {
            return singleReturnOnly && returnsImmediately(node)
                ? config_1.USE_ARROW_WHEN_SINGLE_RETURN
                : config_1.USE_ARROW_WHEN_FUNCTION;
        };
        return {
            'ExportDefaultDeclaration > FunctionDeclaration': function (node) {
                if (isSafeTransformation(node)) {
                    context.report({
                        fix: function (fixer) {
                            return fixer.replaceText(node, writeArrowFunction(node) + ';');
                        },
                        message: getMessage(node),
                        node: node,
                    });
                }
            },
            ':matches(ClassProperty, MethodDefinition, Property)[key.name][value.type="FunctionExpression"][kind!=/^(get|set)$/]': function (node) {
                var propName = node.key.name;
                var functionNode = node.value;
                if (isSafeTransformation(functionNode) &&
                    (!isWithinClassBody(functionNode) || classPropertiesAllowed)) {
                    context.report({
                        fix: function (fixer) {
                            return fixer.replaceText(node, isWithinClassBody(node)
                                ? propName + " = " + writeArrowFunction(functionNode) + ";"
                                : propName + ": " + writeArrowFunction(functionNode));
                        },
                        message: getMessage(functionNode),
                        node: functionNode,
                    });
                }
            },
            'ArrowFunctionExpression[body.type!="BlockStatement"]': function (node) {
                if (returnStyle === 'explicit' && isSafeTransformation(node)) {
                    context.report({
                        fix: function (fixer) { return fixer.replaceText(node, writeArrowFunction(node)); },
                        message: config_1.USE_EXPLICIT,
                        node: node,
                    });
                }
            },
            'ArrowFunctionExpression[body.body.length=1][body.body.0.type="ReturnStatement"]': function (node) {
                if (returnStyle === 'implicit' && isSafeTransformation(node)) {
                    context.report({
                        fix: function (fixer) { return fixer.replaceText(node, writeArrowFunction(node)); },
                        message: config_1.USE_IMPLICIT,
                        node: node,
                    });
                }
            },
            'FunctionExpression[parent.type!=/^(ClassProperty|MethodDefinition|Property)$/]': function (node) {
                if (isSafeTransformation(node)) {
                    context.report({
                        fix: function (fixer) { return fixer.replaceText(node, writeArrowFunction(node)); },
                        message: getMessage(node),
                        node: node,
                    });
                }
            },
            'FunctionDeclaration[parent.type!="ExportDefaultDeclaration"]': function (node) {
                if (isSafeTransformation(node)) {
                    context.report({
                        fix: function (fixer) {
                            return fixer.replaceText(node, writeArrowConstant(node) + ';');
                        },
                        message: getMessage(node),
                        node: node,
                    });
                }
            },
        };
    },
};
